/*
 * Copyright (c) 2017-2019, FusionAuth, All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific
 * language governing permissions and limitations under the License.
 */

package io.fusionauth.der;

import io.fusionauth.pem.domain.PEM;
import org.testng.annotations.Test;

import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertTrue;

/**
 * @author Daniel DeGroff
 */
public class DerInputStreamTest {
  @Test
  public void bitstring_with_ignored_bits() throws Exception {
    // Assert the example encoding provided here:
    // https://security.stackexchange.com/questions/10392/parsing-the-id-ce-keyusage-extension-in-x-509-certs

    // Without ignoring the last bit the result of 0x06 would be 0b00000110 (8 bits)
    assertEquals(0x06, 0b00000110);

    // Chop 1 to 4 bits off
    for (int i : new int[]{0x01, 0x02, 0x03, 0x04}) {
      byte[] bytes = new byte[]{
          0x03,     // Tag: Bit String
          0x02,     // Length: 2 bytes to follow
          (byte) i, // Leading byte: Ignore the last bit (0b00000001)
          0x06      // The byte to encode too bits
      };

      DerInputStream is = new DerInputStream(bytes);
      DerValue value = is.readDerValue();
      assertTrue(value.tag.is(Tag.BitString));
      assertEquals(value.getLength(), 2);

      // But because we ignore the last bit as instructed, it will be 8 bits - n (where n = i)
      assertEquals(value.getBitString().length(), 8 - i);
      assertEquals(value.getBitString(), "00000110".substring(0, 8 - i));
    }
  }

  @Test
  public void ec_private_key_notEncapsulated() throws Exception {
    // These keys are not yet encapsulated in a PKCS#8 PEM
    List<String> files = Arrays.asList(
        "ec_private_prime256v1_p_256_openssl.pem",
        "ec_private_secp384r1_p_384_openssl.pem",
        "ec_private_secp521r1_p_512_openssl.pem");

    for (int i = 0; i < files.size(); i++) {
      String rawPEM = new String(Files.readAllBytes(Paths.get("src/test/resources/" + files.get(i))));

      int startIndex = rawPEM.indexOf(PEM.EC_PRIVATE_KEY_PREFIX);
      int endIndex = rawPEM.indexOf(PEM.EC_PRIVATE_KEY_SUFFIX);

      String base64 = rawPEM.substring(startIndex + PEM.EC_PRIVATE_KEY_PREFIX.length(), endIndex).replaceAll("\\s+", "");
      byte[] bytes = Base64.getDecoder().decode(base64);

      // This is an EC private key, encapsulate it in a PKCS#8 format to be compatible with the Java key factory
      //
      // EC Private key
      // ------------------------------------------------------
      // PrivateKeyInfo ::= SEQUENCE {
      //   version         Version,
      //   PrivateKey      OCTET STRING
      //   [0]             Context Specific
      //        curve       - OBJECT IDENTIFIER
      //   [1]             Context Specific
      //        -           - BIT STRING
      // }

      DerInputStream is = new DerInputStream(bytes);
      DerValue[] sequence = is.getSequence();

      assertEquals(sequence.length, 4);
      assertEquals(sequence[0].tag.value, Tag.Integer);
      assertEquals(sequence[1].tag.value, Tag.OctetString);
      assertEquals(sequence[2].tag.value, 0);
      assertEquals(sequence[2].tag.tagClass, TagClass.ContextSpecific);
      assertEquals(sequence[3].tag.value, 1);
      assertEquals(sequence[3].tag.tagClass, TagClass.ContextSpecific);

      // Version 1
      assertEquals(sequence[0].getBigInteger().intValue(), 1);

      // Curve OID
      if (i == 0) {
        assertEquals(sequence[2].getOID().decode(), ObjectIdentifier.ECDSA_P256);
      } else if (i == 1) {
        assertEquals(sequence[2].getOID().decode(), ObjectIdentifier.ECDSA_P384);
      } else {
        assertEquals(sequence[2].getOID().decode(), ObjectIdentifier.ECDSA_P521);
      }
    }
  }

  @Test
  public void ec_public_key_256() throws Exception {
    PEM pem = PEM.decode(Paths.get("src/test/resources/ec_public_key_p_256.pem"));

    // DER Encoded Public Key Format SubjectPublicKeyInfo
    // ------------------------------------------------------
    // SubjectPublicKeyInfo ::= SEQUENCE {
    //   algorithm         AlgorithmIdentifier,
    //   subjectPublicKey  BIT STRING
    // }
    //
    // AlgorithmIdentifier ::= SEQUENCE {
    //   algorithm       OBJECT IDENTIFIER,
    //   parameters      ANY DEFINED BY algorithm OPTIONAL
    // }

    DerInputStream is = new DerInputStream(pem.publicKey.getEncoded());
    DerValue[] sequence = is.getSequence();

    assertEquals(sequence.length, 2);
    assertTrue(sequence[0].tag.is(Tag.Sequence));

    DerInputStream nested = new DerInputStream(sequence[0].toByteArray());
    ObjectIdentifier encryptionOID = nested.getOID();
    assertEquals(encryptionOID.toString(), ObjectIdentifier.EC_ENCRYPTION);
    ObjectIdentifier curveOID = nested.getOID();
    assertEquals(curveOID.toString(), ObjectIdentifier.ECDSA_P256);

    // Account for all of the bytes in the first sequence
    assertEquals(encryptionOID.value.length + curveOID.value.length + 4, nested.length);

    // The last portion of the key is the public key in a bit string
    assertEquals(sequence[1].tag.value, Tag.BitString);
    assertEquals(sequence[1].getBitString().length(), 520);
    assertEquals(sequence[1].getBitString(), "0000010000110100100001011010100110110000100001111001101011010010101011001100000111011011011001001011010010111010110100001100011011110001110100011000010101100010101000010000100010011110111111110100001011101101100110000001100110111000101010110111000010001110011110101000100101010001010001011011010100111000001000100010001001111100111111110111111000111000011111101001111101010110001010100110111000100110111001110101000110111101101011111011010000010110010001011011001101111100000011111010010100001100001101111001100111001101");
  }

  @Test
  public void ec_public_key_384() throws Exception {
    PEM pem = PEM.decode(Paths.get("src/test/resources/ec_public_key_p_384.pem"));

    // DER Encoded Public Key Format SubjectPublicKeyInfo
    // ------------------------------------------------------
    // SubjectPublicKeyInfo ::= SEQUENCE {
    //   algorithm         AlgorithmIdentifier,
    //   subjectPublicKey  BIT STRING
    // }
    //
    // AlgorithmIdentifier ::= SEQUENCE {
    //   algorithm       OBJECT IDENTIFIER,
    //   parameters      ANY DEFINED BY algorithm OPTIONAL
    // }

    DerInputStream is = new DerInputStream(pem.publicKey.getEncoded());
    DerValue[] sequence = is.getSequence();

    assertEquals(sequence.length, 2);
    assertTrue(sequence[0].tag.is(Tag.Sequence));

    DerInputStream nested = new DerInputStream(sequence[0].toByteArray());
    ObjectIdentifier encryptionOID = nested.getOID();
    assertEquals(encryptionOID.toString(), ObjectIdentifier.EC_ENCRYPTION);
    ObjectIdentifier curveOID = nested.getOID();
    assertEquals(curveOID.toString(), ObjectIdentifier.ECDSA_P384);

    // Account for all of the bytes in the first sequence
    assertEquals(encryptionOID.value.length + curveOID.value.length + 4, nested.length);

    // The last portion of the key is the public key in a bit string
    assertEquals(sequence[1].tag.value, Tag.BitString);
    assertEquals(sequence[1].getBitString().length(), 776);
    assertEquals(sequence[1].getBitString(), "00000100110011111010100100110001100111000000111111000111011001001111111100101101111101000101111101010111000001000111111110011000011000100000100000011101010000100010111000000101101011011100101110100001010010000100100101101110110100100110110001011100010001001000010011001000101110001001000110111011011000100000101101010010010110011000010100100110000001000111001101101111111101001001110100111110101110000011010010100001100011001110000111010001011110101001000001111001100101100101111100000001111100000010101100111110000000111000100100110001011011101100111100011100100101111110001111011111000100110100100000000000101101101010000100111011110110000010011101100001011011011011111001011001001111001101011100100101101011110011100010111010101011010010011111110001110101010011101110010100");
  }

  @Test
  public void ec_public_key_521() throws Exception {
    PEM pem = PEM.decode(Paths.get("src/test/resources/ec_public_key_p_521.pem"));

    // DER Encoded Public Key Format SubjectPublicKeyInfo
    // ------------------------------------------------------
    // SubjectPublicKeyInfo ::= SEQUENCE {
    //   algorithm         AlgorithmIdentifier,
    //   subjectPublicKey  BIT STRING
    // }
    //
    // AlgorithmIdentifier ::= SEQUENCE {
    //   algorithm       OBJECT IDENTIFIER,
    //   parameters      ANY DEFINED BY algorithm OPTIONAL
    // }

    DerInputStream is = new DerInputStream(pem.publicKey.getEncoded());
    DerValue[] sequence = is.getSequence();

    assertEquals(sequence.length, 2);
    assertTrue(sequence[0].tag.is(Tag.Sequence));

    DerInputStream nested = new DerInputStream(sequence[0].toByteArray());
    ObjectIdentifier encryptionOID = nested.getOID();
    assertEquals(encryptionOID.toString(), ObjectIdentifier.EC_ENCRYPTION);
    ObjectIdentifier curveOID = nested.getOID();
    assertEquals(curveOID.toString(), ObjectIdentifier.ECDSA_P521);

    // Account for all of the bytes in the first sequence
    assertEquals(encryptionOID.value.length + curveOID.value.length + 4, nested.length);

    // The last portion of the key is the public key in a bit string
    assertEquals(sequence[1].tag.value, Tag.BitString);
    assertEquals(sequence[1].getBitString().length(), 1064);
    assertEquals(sequence[1].getBitString(), "00000100000000000000010010001010101101001101011001100111111110110000010010000111111001100000001111101000100100110001110100010010010001110110010001111101101011101001101111111100000110110011011111111100101001100110010010010100001001111100011101000001101111101011101100010010001011010010001011111110100111010010001100101101010000000010111001100101101100100110100000010011001010101001110010011101111100101001110111010110011001000101100100100111110111000001100100010100100000011000001111101011001101110001101011010001111100010110011100110010000000011110100001011011001100011000011111011100010001001010111100010001011111011110000111000110110110010111000111110010010110001100110100001111101110000100010101010110011100001010000001110111111001001101110000010111100001011100011110111101100110011110001101110010100011101100100000001001000000011001001111000001011010000010011111010001111101000101101010000011011001101011001111000010011100111000000000010001101001110000111101010100001101100110100101011000101100011001111110100011110111110000000111100110110001101110111110001100");
  }

  @Test
  public void rsa_public_key_2048() throws Exception {
    // PKCS#1 PEM format, but the public key extracted will contain the encapsulated encoding
    PEM pem = PEM.decode(Paths.get("src/test/resources/rsa_public_key_2048.pem"));

    // DER Encoded Public Key Format SubjectPublicKeyInfo
    // ------------------------------------------------------
    // SubjectPublicKeyInfo ::= SEQUENCE {
    //   algorithm         AlgorithmIdentifier,
    //   subjectPublicKey  BIT STRING
    // }
    //
    // AlgorithmIdentifier ::= SEQUENCE {
    //   algorithm       OBJECT IDENTIFIER,
    //   parameters      ANY DEFINED BY algorithm OPTIONAL
    // }

    DerInputStream is = new DerInputStream(pem.publicKey.getEncoded());
    DerValue[] sequence = is.getSequence();

    assertEquals(sequence.length, 2);
    assertTrue(sequence[0].tag.is(Tag.Sequence));

    DerInputStream nested = new DerInputStream(sequence[0].toByteArray());
    ObjectIdentifier encryptionOID = nested.getOID();
    assertEquals(encryptionOID.toString(), ObjectIdentifier.RSA_ENCRYPTION);

    // The RSA key has a NULL following the encryption Object Identifier, NULL tag followed by 0
    Tag tag = new Tag(nested.data.read());
    assertEquals(tag.value, Tag.Null);
    assertEquals(nested.data.read(), 0);

    // Account for all of the bytes in the first sequence
    assertEquals(encryptionOID.value.length + 4, nested.length);

    // The last portion of the key is the public key in a bit string
    assertEquals(sequence[1].tag.value, Tag.BitString);
    assertEquals(sequence[1].getBitString().length(), 2208);
    assertEquals(sequence[1].getBitString(), "001100001000001000000001000100000000001010000010000000010000011100000010111001110010000101101111001011011110010101100100101000011101101110001000000001100110101101001110010000100110000111101111110110000100100110111110000000001011011101011011101101001001001101001001001011100011001011101001010001100100010010000110000111000111000111011110011011011101011011011010000100001111110010101000001011001011000011101000000100010011001010101101110010001000111110100111000010100010111011111010111110011001111000001100110111101110101000000101001010010011011001000000101010101001001101001110001110010110000011100100111110010001111111110000111011100011001111110001110111110100011100000010010100000101001101010000001110000011100010100111110111110010101000101100010100111000010111011000010000010101011001011001001110001000010101101101000011000000000000111111011010101000111000001100111000100010111111110101011000001000000000110011011010101110010001011100010000111111000000011010001000010111110111101011011001011100000010100000111011000010001001001010011010010101110101111000001110111011101101010010100011100100000011111001101110011101001110001011010011001111011110100000110110111011100010101010011011111111100011110010010010111111110111010010100011110100001000110010001000010000101110110011001100010011110110001110110001111110100000011111011111011001011000010011101111010100011100001110101010000110110001000110001101010100110101110101110100001100100111110011010101111011100111001111000100000100110111110001010000010010011001101001000111111010000010011001011010101001101011101011011110010101101000111000110111011000101000000000110100111110111010000000100100111111101100111110101101100000100111111100111100001011110110001010010101000001111101010111011100010111010110101100001110100110101001101110111110010100010101000010010010110010000110100010000100100111011111110101010110101101010110100100010010110101001001000110011000100111010111011010011110101001001100010110111010110010100101011100101100000010100101111010010000111011001010001001110011011100100111110110001110100011101001100100100011101110100011011100100001110101000010011011001011011101101110100110011100100100100001001111111101010000001000000011000000010000000000000001");
  }

  @Test
  public void rsa_public_key_3072() throws Exception {
    PEM pem = PEM.decode(Paths.get("src/test/resources/rsa_public_key_3072.pem"));

    // DER Encoded Public Key Format SubjectPublicKeyInfo
    // ------------------------------------------------------
    // SubjectPublicKeyInfo ::= SEQUENCE {
    //   algorithm         AlgorithmIdentifier,
    //   subjectPublicKey  BIT STRING
    // }
    //
    // AlgorithmIdentifier ::= SEQUENCE {
    //   algorithm       OBJECT IDENTIFIER,
    //   parameters      ANY DEFINED BY algorithm OPTIONAL
    // }

    DerInputStream is = new DerInputStream(pem.publicKey.getEncoded());
    DerValue[] sequence = is.getSequence();

    assertEquals(sequence.length, 2);
    assertTrue(sequence[0].tag.is(Tag.Sequence));

    DerInputStream nested = new DerInputStream(sequence[0].toByteArray());

    ObjectIdentifier encryptionOID = nested.getOID();
    assertEquals(encryptionOID.toString(), ObjectIdentifier.RSA_ENCRYPTION);

    // The RSA key has a NULL following the encryption Object Identifier, NULL tag followed by 0
    Tag tag = new Tag(nested.data.read());
    assertEquals(tag.value, Tag.Null);
    assertEquals(nested.data.read(), 0);

    // Account for all of the bytes in the first sequence
    assertEquals(encryptionOID.value.length + 4, nested.length);

    // The last portion of the key is the public key in a bit string
    assertEquals(sequence[1].tag.value, Tag.BitString);
    assertEquals(sequence[1].getBitString().length(), 3184);
    assertEquals(sequence[1].getBitString(), "0011000010000010000000011000101000000010100000100000000110000001000000001011010001001111000000101011011010000010000011000101001110010101000011011101111110011011101011000010000000101000110011000001010010001001100011010001100010100000110111010010010110011001001110101111111110011101101000001101011000111001000000011111100000000101100010001001110111001111110001011010000011101000111100110001000111001100001110100110000010001111010100111110101110100000000110010011100010000000000111100110010101110000010100100100110100010111110010111110001101011101001100010000010110101110001010001011110100010101011011111111101111011100110001010101010110000011000001000101100111111111001010000000110000100011011101000010000110100100110111110001001101100101001000100110101100111010100110000100011110101100001111010010101000100110010101111000111011001101111011010110010001001101010010111011010110000011101111100001100111001101010000111000011011001111010010111010000111010101011101100000010101010010110010100101100110110110101011111110111010110010111000001000010001011001010111010101101100110010110101000110001111110010110001100001110111001000011001100111000001001000100100011110010111001001001011010000101110111110010101010001010000111010000101101100111101111011110110110001011101011010011000000111001010110111100010000111100011010110001000011111011110010101000001001000111111100001011110101110000001000101110001101001100001111000001111001000100110000011111010110111001001111000100110100001000011010001100011101111010110111101111111010000011110111010001011001100001001101101001100101111101001110111011000001001011001110011111010011100110110111100000111111111011101101100000000100100111100001111001101000001011111100101110100110100010010110000101111011000001111000100110011010110010111101001001000010000000110010100111011001010111011101001111100011101100011010111000100111100100100011011100100111011101010010101000100110001001000001001111110100111101010000000011011111110001011111111001110010101101111000111111000001011000010101000000111110010000111100010000111100011000000001110011100101111010101011111110000001110110001010111101101010101011011011101100000000010011101111011000010010101110001000110111111101101110110101100010000011110000111101111100000001001010001110111010001011101101111000100000000010100110110101011100010100100101011111001001111000111010001110011110010000111110111100110101000110100001011010100100011101000000111011111110111101110110110100110100011110000110101000100000000001101110111100000000010001111001011111110101011011000001001011011001011001100000000000111001000000110100110011011100011101111110011101100001000011001101101111011000001001111111111010110101110111101100000100011100111001101110001001111010001111011010001010110001100111011111100111000100011001000101111110101111011000100000011000110001011001101001100111100001000100111111010111111110111000010110010010000000010001101110000011101001100101111111110111001000000111000001110011010000011000101011000110000111100110000110000001100010100111010110000010100010000111111101010011000010111111011011111010100000011011010110010111101100101101000000100110001001110101011011111011111111000110000001000000011000000010000000000000001");
  }

  @Test
  public void rsa_public_key_4096() throws Exception {
    PEM pem = PEM.decode(Paths.get("src/test/resources/rsa_public_key_4096.pem"));

    // DER Encoded Public Key Format SubjectPublicKeyInfo
    // ------------------------------------------------------
    // SubjectPublicKeyInfo ::= SEQUENCE {
    //   algorithm         AlgorithmIdentifier,
    //   subjectPublicKey  BIT STRING
    // }
    //
    // AlgorithmIdentifier ::= SEQUENCE {
    //   algorithm       OBJECT IDENTIFIER,
    //   parameters      ANY DEFINED BY algorithm OPTIONAL
    // }

    DerInputStream is = new DerInputStream(pem.publicKey.getEncoded());
    DerValue[] sequence = is.getSequence();

    assertEquals(sequence.length, 2);
    assertTrue(sequence[0].tag.is(Tag.Sequence));

    DerInputStream nested = new DerInputStream(sequence[0].toByteArray());

    ObjectIdentifier encryptionOID = nested.getOID();
    assertEquals(encryptionOID.toString(), ObjectIdentifier.RSA_ENCRYPTION);

    // The RSA key has a NULL following the encryption Object Identifier, NULL tag followed by 0
    Tag tag = new Tag(nested.data.read());
    assertEquals(tag.value, Tag.Null);
    assertEquals(nested.data.read(), 0);

    // Account for all of the bytes in the first sequence
    assertEquals(encryptionOID.value.length + 4, nested.length);

    // The last portion of the key is the public key in a bit string
    assertEquals(sequence[1].tag.value, Tag.BitString);
    assertEquals(sequence[1].getBitString().length(), 4208);
    assertEquals(sequence[1].getBitString(), "00110000100000100000001000001010000000101000001000000010000000010000000011000010111100010100111001001110010001111111011110100011100010010001101110001000001110010101111001110100100110001000110101010001100110001101010011000001110111100101000110000001011010011111100110100101110010101110000000001001011001100011111100011000001101000101010111011110010011000011001100100110111110011111011100111010100010101111011010111000110011000000101010000000010100100000000100000010000110000110100101110101111111000111010000101000000100011011010110010110000001111011001001100100110111001101010010001011100101000011110010011001100100011000111011000111001010111001010000100000111000001000010010001011000101000001000101101011110110010000000100101111011110101010100110100001111010010001011101010101011111101001110001100001111101101110001000010101000000010101100111001100011000111000111000010000010100010100101110101110010100011110110010100101010101110101110111011001000111000110001111011000011101011111110001000100100011100000100001111001011000010001010000010011111001100100011110011001011001001110110110110110001001110111000100101001101111011010101010010011001000011011110101100110101010010101000010110010101100101011110010000011001110010111100110010000001101001011000100010101010100010001111010001010110000100100100100010101001000101101001010101001001010011011000101000110101110010000010000011010011001000010001011000111010100010001101100111100000011100101111010110101100100001001000011001111101001011110011010100101010010010110010111001010010100001110100000100111011011000100111010000011010000000001110000000001110110100010111111000000011011000100000000111110110001101110001001101000000100101100101111001010101010000110110010000111101100100100100001000000110110000100111110111101100001110100011011000111000001011000101101101111111001000110111010100110011010010001001010011110000110110100001011010111010000110101001101000000011101111111001111010011111001011010000001110110100011001010110010100100001101111011011011101011001000111011101000011001101001010111010000010010010111110010000110111000111110001000011110001101100111000101101100110000000000011010101101100010001001010100100110011000100010101100000110110100011001111110100111010011010001011101010010001011101100010111010111110111011001000110011100001011001101101011010101001000100010010101111100111100001100100101011101001100110001001110001110000111011000101101001010111010100011100101101000010110001011011011111010000011110011010001010110001111110101010111000000010000000001010101011111110000001000100101000011101101100111101110100000110011000101001100010001111011100111110111111000001010010101010111100100110000001110000001010111100001011111011111001001111100011101010000111110100000101001000101010011000011111011011001110101001001010100001000111000111000100110001101011001101110000000010110110010010101010110011111101000001100011101101011000001100101010011001000111100111110101010011111110111111011110000110111011000111100000000011010101000110000110101000000010000100101011011000011001111100101001110010100100110111101000000000001000111101100110100101111100111011011111101110011010100010000111001101111010000010110001101101010001011110111100100001111000100011001000100111101110110101001011110010101100100000011010110110000011010101010101111010001001010011111111001001101001010101011110100100111010011110100100010000011100010000001010111110001111001011101110000101101111010000010111011110010100001000011001010100001010010010010111111001101010101101001011011000110100011001001011110000000010100000111000110111001011000001011100011010101100100010100101110110101000010110100111110000101110111001100000000111001111000011000100010011010101101101111111001111111010101011001010110011110110011111101100110111011000110101010101110100010111000001010000100011100111101110100101111010000111110001100100110011011110000010000100010010110111101001110101111111000101111111011101001100110000000111111110000000011000011011111010000110000100101010000110111111001010010010100110100101000011100110101100101001111101110111011111011000001001101000100010111111011111101001001100001011001100110111111110110101101001110110010111000100011111000100111101111010000001000000011000000010000000000000001");
  }
}
